package Cluedo.API;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Image;
import java.io.File;

import javax.swing.ImageIcon;
import javax.swing.JComponent;

/**
 * CardImage provides a way of showing a card even when the image may not exist
 * on disk. It can be added to any Container object and will resize itself
 * accordingly to the file specified.
 * 
 * @author Alex Ellis
 */
public class CardImage extends JComponent {
	/**
	 * The image behind the card
	 */
	Image image = null;
	
	/**
	 * The image behind the card for display
	 */
	ImageIcon imageIcon = null;
	
	/**
	 * Dimension of the card got from the image variable
	 */
	Dimension dimensions = new Dimension(84, 133);
	
	static final Dimension defaultDimensions = new Dimension(84, 133);
	
	/**
	 * The name of the card
	 */
	String name = null;
	
	/**
	 * @param icon An image icon to replace the current one
	 * @param name the name of the card
	 *
	 */
	public void changeCard(ImageIcon icon, String name) {
	    if(icon == null) {
	        System.out.println("changeCard("+name+")");
	        changeCard(name);
		    resize();
	    }
	    else {
	        this.imageIcon = icon;
	        setupDimensions(icon);
	        this.name = name;
	    }

	}
	
	/**
	 * @param icon Filename of image to replace the current one
	 * @param name the name of the card
	 *
	 */
	public void changeCard(String fileName, String name) {
		this.name = name;
		image = null;
		imageIcon = null;
		loadImage(fileName, name);
		
		if (imageIcon == null || imageIcon.getIconWidth() < 10)
			imageIcon = null;
		
		if (imageIcon != null)
			setupDimensions(imageIcon);
		else
			dimensions = defaultDimensions;
	}
	
	/**
	 * @param name the name of the card
	 *
	 */
	public void changeCard(String name) {
		image = null;
		imageIcon = null;
		this.name = name;
		dimensions = defaultDimensions;
	}
	
	/**
	 * Creates dimensions for the loaded image
	 *
	 */
	private void setupDimensions(ImageIcon icon) {
		// Cast to ImageIcon to Image to get some dimensions data from it
		if(icon!=null) {
			image = icon.getImage();
		
			dimensions = new Dimension(icon.getIconWidth(), icon.getIconHeight());
		}
		else {
			dimensions = defaultDimensions;
			
		}
		
		// Make the image doesn't grow / shrink
		resize();
		
	}
	
	public void resize() {
		setMaximumSize(dimensions);
		setMinimumSize(dimensions);
		setPreferredSize(dimensions);
	}
	/**
	 * Use this if you don't have the ImageIcon cached
	 *
	 * @param fileName Filename of the card's image
	 * @param name the name of the card
	 *
	 */
	public CardImage(String fileName, String name) {
		this.name = name;
		loadImage(fileName, name);
		
		if (imageIcon == null || imageIcon.getIconWidth() < 10)
			imageIcon = null;
		
		if (imageIcon != null)
			setupDimensions(imageIcon);
	}
	
	/**
	 * If you have the ImageIcon already, use this.
	 *
	 * @param imageIcon image for this card
	 * @param name the name of the card
	 *
	 */
	public CardImage(ImageIcon imageIcon, String name) {
		this.name = name;
		this.imageIcon = imageIcon;
		if (imageIcon != null)
			setupDimensions(imageIcon);
	}
	
	/**
	 * Private function loads an image or creates a card with just a caption
	 *
	 */
	private void loadImage(String fileName, String caption) {
		// To get full path to file
		File filePointer = new File(fileName);
		this.name = caption;
		
		try {
			// Load image as ImageIcon
			if (filePointer.exists() == true) {
				imageIcon = new ImageIcon(filePointer.getPath()); // Resolves path to file on disk.
				setupDimensions(imageIcon);
			} else {
				imageIcon = null;
				image = null;
			}
			
		} catch (Exception e) {
		}
	}
	
	/**
	 * Paints either the image or a string in Helvetica, BOLD, size 12
	 */
	public void paintComponent(Graphics g) {
		
		Graphics g2 = (Graphics2D) g;
		if(name.equals(""))
		    return;

		if (imageIcon != null) {
			// The background image
			imageIcon.paintIcon(this, g, 0, 0);
		} else {
			g2.setColor(Color.white);
			g2.fillRect(0, 0, defaultDimensions.width,defaultDimensions.height);
			
			g2.setColor(Color.black);
			g2.drawRect(0,0,defaultDimensions.width-1,defaultDimensions.height-1);

			g2.setFont(new Font("Helvetica", Font.BOLD, 12));
			g2.drawString(name, 5, dimensions.height / 2);
		}
	}
}